% Script file fault_slip_main
%
% Computes and plots depletion-induced stresses and fault slip along an inclined fault.
%
% Version 2.11, April 2022.
%
% Jan Dirk Jansen, Delft University of Technology (TU Delft)
% j.d.jansen@tudelft.nl
%
% See the ReadMe.txt file for info.
%

clear variables
close all

% ---------------------------------------------------------------------------------------------
% Set input parameters:
% ---------------------------------------------------------------------------------------------
% Parameter set for Example 1 from paper:
a = 75; % see Figure 1, m
alpha = 0.9; % Biot's coefficient, -
b = 150; % see Figure 1, m
delta_c = 0.02; % critical slip distance for slip weakening 0, m
D_center = 3500; % reservoir depth @ y = 0, m
flag_fric = 1; % flag to indicate friction law 
    % 0 = no friction
    % 1 = constant friction
    % 2 = slip-weakening friction
g = 9.81; % acceleration of gravity, m/s^2
G = 6500e6; % shear modulus, Pa
kappa = 0; % cohesion, -
K_eff = 0.5; % initial effective vertical/horizontal stress ratio, -
mu_dyn = 0.20; % dynamic friction coefficient, -
mu_st = 0.52; % static friction coefficient, -
nu = 0.15; % Poisson's ratio, -
p_0_center = 35e6; % initial pressure @ y = 0, Pa
phi = 0.15; % porosity, -
rho_f = 1020; % fluid density, kg/m^3
rho_s = 2650; % solid density, kg/m^3
sigma_v_0 = 0; % initial vertical stress @ y = 0, Pa. If set equal to zero, the stress is ...
    % computed from the weigth of the overburden; see function fault_slip_sigma.m.
theta = 70*pi/180; % dip angle, rad

% ---------------------------------------------------------------------------------------------
% Set program flow flags and computational parameters:
% ---------------------------------------------------------------------------------------------
f_damp = 0.5; % damping factor for iterative update of slip-weakening and coupled ...
    % simulation (only relevant for flag_cheb == 1)
flag_accu = 1; % flag to set level of accuracy in computation; espcially relevant for ...
    % slip-weakening friction where accurate determination of the nucleation pressure is ...
    % time consuming
    % 1 = reasonably fast and reasonably accurate (recommended for regular use)
    % 2 = accurate but slow (minutes for flag_cheb = 1; tens of minutes flag_cheb = 0 ...
    %     especially in combination with flag_fric = 1) 
flag_cheb = 0; % flag to trigger use of Chebyshev polynomials for numerical integration
    % 0 = numerical integration with near-uniform staggered grids for y and xi (for diagnostics
    %     and to producee snap shot plots)
    % 1 = semi-analytical inversion through integration with Chebyshev polynomials
    %     (recommended for regular use)
flag_check = 0; % flag to trigger graphical check of finite difference and Chebyshev results
    % 0 = check disabled (default) 
    % 1 = check enabled (for diagnostics; causes overwriting of all input from main file; ...
    %     see fault_slip_patch_bounds_check.m)
flag_cont = 1; % flag to force continuation in case of non-convergence of slip-weakening ...
    % and coupled simulation (only relevant for flag_cheb == 1)
    % 0 = continuation disabled (for diagnostics) 
    % 1 = continuation enabled (recommended for regular use)
flag_cor = 1; % flag to force correct sign on slip gradient (only relevant for flag_cheb = 0)
    % 0 = forcing switched off (for diagnostics)
    % 1 = forcing switched on (recommended for regular use) 
flag_decouple = 1; % flag to force decoupling between slip patches (only relevant for ...
    % flag_fric > 1; for flag_fric = 1, coupling is always active.)
    % 0 = coupled simulation; takes a lot more time (for diagnostics)
    % 1 = decoupled simulation; much faster but occasionally inaccurate
    %     (recommended for regular use)
flag_echo = 2; % flag to trigger writing to screen of simulation progress
    % 0 = no echo
    % 1 = limited echo; only info about main simulation components
    % 2 = full echo; info about each simulation step
flag_eig_use = 1; % flag to determine use of eigenvalues (only relevant for flag_fric > 1)
    % 0 = used to check accuracy of simulated slip length (Delta y*_sim ~= Delta y*_eig?) 
    %     (for diagnostics)
    % 1 = used for seismicity detection (stops simulation when Delta y*_sim > Delta y*_eig)
    %     (first option for regular use)
    % 2 = numerical derivative used for seismicity detection (stops simulation when ...
    %     (dp/dDelta y) drops below tol_deriv); requires flag_lag = 0  
    %     (second option for regular use)
tol_deriv = 1e-6; % simulation stopping criterion in terms of (almost) zero scaled ...
    % numerical derivative (dp/dDelta y)*(Delta_y/A), where A = G / (2*pi*(1-nu)), - 
flag_lag = 0; % flag to trigger lagging of eigen problem computation with one presure step;
    % (only relevant for flag_fric > 1)
    % 0 = lagging switched off (recommended for regular use)
    % 1 = lagging switched on (for diagnostics) 
    % automatically switched on in case of convergence failure in fault_slip_delta(_cheb).m
flag_M = 0; % flag to compute seismic moment (can be time consuming)
    % 0 = computation switched off
    % 1 = computation switched on
flag_paper = 0; % flag to trigger plotting of figure for paper. Warning: may take quite ...
    % some time, depending on which plots you choose; see fault_slip_patch_bounds_paper.m ...
    % which contains several further flags to select the various plots.  
    % 0 = default plots
    % 1 = plot for paper; causes overwriting of all input from main file
flag_reg = 1; % flag to control regularization of slip-dependent friction coefficient and ...
    % Coulomb stresses:
    % 0 = regularization switched off (for diagnostics)
    % 1 = regularization switched on (recommended for regular use)
eps_reg = 0; % regularization parameter for slip-dependent friction coefficient, m
eta_reg = 0.10; % regularization parameter for Coulomb stresses, m
flag_top_only = 1; % flag to force occurrence of seismicity in top patch only (only ...
    % relevant for flag_cheb == 1)  
    % 0 = forcing switched off (for diagnostics)
    % 1 = forcing switched on (recommended for regular use)
    
% Computational parameters for patch size computation:
if flag_fric == 0 % zero friction
    Delta_y_int = 0; % dummy
    tol_rel_coupled = 0; % dummy
end
if flag_fric == 1 % constant friction
    switch flag_accu
        case 1
            Delta_y_int = 0.10; 
            n_max_int = 2000; 
            cheb_terms = 200; 
            tol_rel_coupled = 1e-6; 
        case 2
            Delta_y_int = 0.05; 
            n_max_int = 5000; 
            cheb_terms = 500; 
            tol_rel_coupled = 1e-8; 
    end
end
if flag_fric == 2 % slip-weakening friction
    switch flag_accu
        case 1
            Delta_y_int = 0.10; 
            n_max_int = 2000; 
            cheb_terms = 100; 
            tol_rel_coupled = 1e-8; 
        case 2
            Delta_y_int = 0.01; 
            n_max_int = 5000; 
            cheb_terms = 500;
            tol_rel_coupled = 1e-10; 
    end
end
n_min_int = 100;  % min. required total nr. of integr. pnts, -. Only relevant for flag_cheb = 0
max_iter_coupled = 500; % maximum allowed number of iterations for patch size computation

% Computational parameters for pressure stepping:
p_max_bounds = -12e6; % highest incremental pressure (negative for production), Pa
p_min_bounds = -30e6; % lowest incremental pressure (negative for production), Pa
Delta_p_init = -0.05e6; % initial pressure step size (negative for production), Pa
Delta_p_max = -1e6; % maximum allowed pressure step size (negative for production), Pa
if flag_fric == 0 % zero friction
    Delta_p_min = 0; % dummy
    Delta_y_target = 0; % dummy
end
if flag_fric == 1 % constant friction
    switch flag_accu
        case 1
            Delta_p_min = -1e3;
            Delta_y_target = 0.50;
        case 2
            Delta_p_min = -1e5;
            Delta_y_target = 0.50;
    end
end
if flag_fric == 2 % slip-weakening friction
    switch flag_accu
        case 1
            Delta_p_min = -100;
            Delta_y_target = 0.10;
        case 2
            Delta_p_min = -50;
            Delta_y_target = 0.05;
    end
end
small_patch = 0.05; % minimum patch size to trigger slip computation during pr. stepping, m

% ---------------------------------------------------------------------------------------------
% Set plotting flags and parameters:
% ---------------------------------------------------------------------------------------------
% Plotting flags:
flag_plot_cap = 0; % flag for plot captions (black) and plot file location captions (red) 
    % 0 = no captions
    % 1 = with captions
flag_plot_diag = 1; % flag for diagnostic plots:
    % 0 = plotting switched off
    % 1 = plotting switched on
flag_plot_eig = 1; % flag for eigenvalue plot (only relevant for slip-weakening friction):
    % 0 = plotting switched off
    % 1 = plotting switched on
flag_plot_fields = 1; % flag for field plots (slip, slip gradient etc. as function of p and y):
    % 0 = plotting switched off
    % 1 = plotting switched on   
flag_plot_snap = 1; % flag for snapshot plots:
    % 0 = plotting switched off
    % 1 = plotting switched on   
% Parameters for snapshot plots of stresses, slip etc.:
    % Note 1: all comments refer to parameter settings for example 1 from the paper.
    % Note 2: different plots are created for zero, constant or slip-weakening ...
    % friction. Moreover for p_inc_02, additional plots are created in case of ...
    % slip-weakening friction. See fault_slip_patch_bounds.m for further info.  
if flag_fric == 0 % zero friction
    p_inc = -25e6;
end
if flag_fric == 0 % zero friction
    p_inc_01 = 0; % dummy
    p_inc_02 = 0;
    p_inc_03 = 0;
    p_inc_04 = 0;
    p_inc_05 = 0;
end
if flag_fric == 1 % constant friction
    p_inc_01 = -24e6; % incremental pressure at which to take a snapshot, Pa
    p_inc_02 = -25e6;
    p_inc_03 = -26e6;
    p_inc_04 = -27e6;
    p_inc_05 = -28e6;
end
if flag_fric == 2 % slip-weakening friction
    p_inc_01 = -14e6; % incremental pressure at which to take a snapshot, Pa
    p_inc_02 = -15e6;
    p_inc_03 = -16e6;
    p_inc_04 = -17e6;
    p_inc_05 = -17.3e6;
end
% Minimum and maximum y values for various plots:
y_plot_max_delta_1 =  100; % covers both slip patches (in detail), m
y_plot_min_delta_1 = -100; % covers both slip patches (in detail), m
y_plot_max_delta_2 =   80; % covers top patch only (further detail), m
y_plot_min_delta_2 =   60; % covers top patch only (further detail), m
y_plot_max_sigma   =  250; % covers both slip patches (overview), m
y_plot_min_sigma   = -250; % covers both slip patches (overview), m
y_plot_n_pt = 2000; % number of points between y_plot_min_... and y_plot_max_...
    % (only relevant for some plots) 
y_plot_max_bounds =  100; % covers both slip patches (in detail), m
y_plot_min_bounds = -100; % covers both slip patches (in detail), m

% Colors for line plots:
blue = [0/255 128/255 255/255];
gray = [235/255 235/255 235/255];
green = [0/255 204/255 0/255];
orange = [255/255 185/255 0/255];
colors = [blue; gray; green; orange];

% ---------------------------------------------------------------------------------------------
% End of input. Start computing:
% ---------------------------------------------------------------------------------------------

if flag_echo > 0
    tic
end

% ---------------------------------------------------------------------------------------------
% Compute and plot stresses and fault slip for a frictionless vertical fault (only relevant for
% flag_fric == 0):
% ---------------------------------------------------------------------------------------------

if flag_fric == 0 && flag_check ~= 1 && flag_paper ~= 1
    fault_slip_frictionless(a,alpha,b,colors,D_center,eps_reg,eta_reg,flag_paper,...
        flag_reg,g,G,kappa,K_eff,nu,p_0_center,p_inc,phi,rho_f,rho_s,sigma_v_0,...
        y_plot_max_sigma,y_plot_min_sigma,y_plot_n_pt);
end

% ---------------------------------------------------------------------------------------------
% Compute and plot Coulomb stress zeros and slip patch boundaries as a function of depletion
% pressure:
% ---------------------------------------------------------------------------------------------

if flag_fric > 0 && flag_paper == 0 && flag_check == 0 % regular simulation 
    fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,D_center,...
        Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
        eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,...
        flag_echo,flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,...
        flag_plot_diag,flag_plot_eig,flag_plot_fields,flag_plot_snap,flag_reg,...
        flag_top_only,g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,...
        nu,p_0_center,p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
        p_max_bounds,p_min_bounds,phi,rho_f,rho_s,sigma_v_0,small_patch,theta,...
        tol_deriv,tol_rel_coupled,...
        y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
        y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
        y_plot_n_pt);
else 
    if flag_check == 1 % produces figures for graphical check of finite difference and ...
            % Chebyshev results
        fault_slip_patch_bounds_check;
    end
    if flag_paper == 1 % produces figures for paper plus a few more
        fault_slip_patch_bounds_paper;
    end
end

if flag_echo > 0
    toc
end
